package ga.view.streaming;

import ga.core.individual.IIndividual;
import ga.view.appstate.SceneState;
import ga.view.factory.EffectsFactory;
import ga.view.input.CamDragListener;
import ga.view.interfaces.MouseListener;
import ga.view.streaming.nodes.PanelNode;
import ga.view.streaming.showroom.CameraSettings;

import com.jme3.app.Application;
import com.jme3.app.state.AppState;
import com.jme3.app.state.AppStateManager;
import com.jme3.math.ColorRGBA;
import com.jme3.math.Vector3f;
import com.jme3.renderer.Camera;
import com.jme3.scene.Node;
import com.jme3.system.AppSettings;

/**
 * This is an app state that shows individuals in the showroom.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class ShowRoomState<T extends IIndividual<T>> extends SceneState {
  private final AppState stateToPause;
  private CamDragListener dragListener;
  private DoubleClickListener doubleClickListener;
  private PanelNode<T> panelNode;

  private AppSettings settings;

  private CameraSettings camSettings;

  /**
   * Instantiates a new show room state.
   * 
   * @param stateToPause
   *          the state to pause
   * @param camSettings
   *          the cam settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ShowRoomState(final AppState stateToPause,
      final CameraSettings camSettings) {
    this.stateToPause = stateToPause;
    this.camSettings = camSettings;
  }

  @Override
  public void initialize(final AppStateManager stateManager,
      final Application app) {
    super.initialize(stateManager, app);
    settings = app.getContext().getSettings();

    viewPort.setBackgroundColor(ColorRGBA.White);

    dragListener = new CamDragListener(cam, inputManager, camSettings);

    dragListener.setEnabled(false);
    doubleClickListener = new DoubleClickListener();
    doubleClickListener.setEnabled(false);
    inputManager.addListener(doubleClickListener, "doubleClick");

    camSettings.configureCamera(cam);

    // TODO this was only for debugging and can be removed soon
    // inputManager.addMapping("Minnaert", new KeyTrigger(KeyInput.KEY_1));
    // inputManager.addMapping("WardIso", new KeyTrigger(KeyInput.KEY_2));
    // inputManager.addMapping("VTangent", new KeyTrigger(KeyInput.KEY_3));
    //
    // inputManager.addListener(new ShadowListener(), "Minnaert", "WardIso",
    // "VTangent");

  }

  /**
   * Sets the cam settings.
   * 
   * @param camSettings
   *          the new cam settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setCamSettings(final CameraSettings camSettings) {
    this.camSettings = camSettings;

    camSettings.configureCamera(cam);
    dragListener.init(camSettings);
  }

  /**
   * Sets the panel node.
   * 
   * @param panelNode
   *          the new panel node
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setPanelNode(final PanelNode<T> panelNode) {
    this.panelNode = panelNode;

    final Camera procCam = panelNode.getProcessor().getCamera();

    cam.setLocation(procCam.getLocation());
    cam.lookAt(camSettings.getLookAtPoint(), Vector3f.UNIT_Y);

    dragListener.init();

    setRootNode(panelNode.getShowRoom());
  }

  /**
   * Sets the root node.
   * 
   * @param node
   *          the new root node
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setRootNode(final Node node) {
    this.rootNode = node;
    rootNode.setName("ShowRoom state RootNode");

    rootNode.updateGeometricState();

    setEnabled(true);

    EffectsFactory.addShadowProcessor(assetManager, settings, viewPort,
        new Vector3f(3.0f, 2.1f, 3.0f));
    EffectsFactory.addLightScatteringProcessor(assetManager, inputManager,
        settings, viewPort, new Vector3f(3.0f, 1.9f, 3.0f));
    EffectsFactory.addSSAOProcessor(assetManager, inputManager, settings,
        viewPort);
  }

  /**
   * Sets the app settings.
   * 
   * @param settings
   *          the new app settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setSettings(final AppSettings settings) {
    this.settings = settings;
  }

  @Override
  public void setEnabled(final boolean enabled) {
    super.setEnabled(enabled);

    if (dragListener != null) {
      dragListener.setEnabled(enabled);
    }
    if (doubleClickListener != null) {
      doubleClickListener.setEnabled(enabled);
    }

    if (viewPort != null) {
      viewPort.setEnabled(enabled);
      if (!enabled) {
        viewPort.clearScenes();
      }
    }
  }

  @Override
  public void stateAttached(final AppStateManager stateManager) {
    if (viewPort != null) {
      viewPort.setEnabled(true);
    }
  }

  @Override
  public void stateDetached(final AppStateManager stateManager) {
    dragListener.setEnabled(false);
    doubleClickListener.setEnabled(false);

    if (viewPort != null) {
      viewPort.clearScenes();
      viewPort.setEnabled(false);
    }
  }

  // private class ShadowListener implements ActionListener {
  //
  // @Override
  // public void onAction(final String name, final boolean keyPressed,
  // final float tpf) {
  // if (keyPressed) {
  // try {
  // final boolean active = !Boolean.valueOf(panelNode.getScene()
  // .getMaterial().getParam(name).getValueAsString());
  // panelNode.getScene().getMaterial().setBoolean(name, active);
  // LOGGER.info(name + " " + (active ? "enabled" : "disabled"));
  // } catch (final Exception e) {
  // LOGGER.log(Level.WARNING, "An error occured", e);
  // }
  // }
  // }
  // }

  /**
   * Mouse listener that exits this state if a double click has been detected.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class DoubleClickListener extends MouseListener {

    @Override
    public void onAction(final String name, final boolean keyPressed,
        final boolean isDoubleClick, final float tpf) {
      if (isDoubleClick && stateToPause != null) {
        ShowRoomState.this.setEnabled(false);
        stateToPause.setEnabled(true);
        final Camera procCam = panelNode.getProcessor().getCamera();

        procCam.setLocation(cam.getLocation());
        procCam.lookAt(Vector3f.ZERO, Vector3f.UNIT_Y);
      }
    }
  }
}
